import tkinter as tk
from tkinter import ttk, messagebox
import subprocess
import os
import sys

class Director64Dashboard:
    def __init__(self, resources_dir):
        self.resources_dir = resources_dir
        self.root = tk.Tk()
        self.root.title("Director64 Dashboard - v4.2")
        
        # Window size and centering
        window_width = 550
        window_height = 420
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()
        center_x = int(screen_width/2 - window_width/2)
        center_y = int(screen_height/2 - window_height/2)
        self.root.geometry(f'{window_width}x{window_height}+{center_x}+{center_y}')
        
        self.root.resizable(False, False)
        
        # Modern Dark Theme
        self.bg_color = "#282828"
        self.fg_color = "#FFFFFF"
        self.root.configure(bg=self.bg_color)
        
        # Style
        self.style = ttk.Style()
        self.style.theme_use('classic')
        self.style.configure('TButton', font=('Helvetica', 11))
        self.style.configure('Action.TButton', font=('Helvetica', 11, 'bold'))
        
        self.setup_ui()
        self.check_status()
        
    def setup_ui(self):
        # Header
        header_frame = tk.Frame(self.root, bg=self.bg_color)
        header_frame.pack(pady=(25, 10), fill="x")
        
        title = tk.Label(header_frame, text="Tonys Director64 v4.1", 
                         font=("Helvetica", 18, "bold"), bg=self.bg_color, fg=self.fg_color)
        title.pack()
        
        # Warning Box
        warning_frame = tk.Frame(self.root, bg="#333333", padx=20, pady=15, highlightthickness=1, highlightbackground="#444444")
        warning_frame.pack(padx=30, pady=10, fill="x")
        
        warning_text = ("WARNING: H.264 video will NOT play.\n"
                        "Please deliver movies in MPEG-4 (.mov).\n"
                        "(The Video Converter tool is included below)")
        warning_lbl = tk.Label(warning_frame, text=warning_text, font=("Helvetica", 12, "bold"),
                              bg="#333333", fg="#FF4B4B", justify="center")
        warning_lbl.pack()
        
        # Status Section
        status_container = tk.Frame(self.root, bg=self.bg_color)
        status_container.pack(padx=50, pady=(15, 0), fill="x")
        
        tk.Label(status_container, text="System Prerequisites:", font=("Helvetica", 12, "bold"),
                 bg=self.bg_color, fg="#AAAAAA").pack(anchor="w", pady=(0, 5))
        
        self.status_vars = {}
        items = [
            ("Homebrew", "brew --version"),
            ("Wine", "wine --version"),
            ("Winetricks", "winetricks --version"),
            ("QuickTime", "qt_check")
        ]
        
        for name, cmd in items:
            row = tk.Frame(status_container, bg=self.bg_color)
            row.pack(fill="x", pady=2)
            
            tk.Label(row, text=f"• {name}:", font=("Helvetica", 11), 
                     bg=self.bg_color, fg=self.fg_color, width=12, anchor="w").pack(side="left")
            
            stat = tk.Label(row, text="Checking...", font=("Helvetica", 11, "bold"), 
                           bg=self.bg_color, fg="#888888")
            stat.pack(side="left")
            self.status_vars[name] = stat
        
        # Bottom Button Bar - All in one row, exactly in order
        btn_container = tk.Frame(self.root, bg=self.bg_color)
        btn_container.pack(side="bottom", fill="x", padx=20, pady=25)
        
        # Order: Quit, Uninstall, Video Converter, Launch
        btn_width = 11
        tk.Button(btn_container, text="Quit", command=sys.exit, width=btn_width, highlightbackground=self.bg_color).pack(side="left", expand=True, padx=2)
        tk.Button(btn_container, text="Uninstall", command=lambda: self.finish("UNINSTALL"), width=btn_width, highlightbackground=self.bg_color).pack(side="left", expand=True, padx=2)
        tk.Button(btn_container, text="Video Converter", command=lambda: self.finish("CONVERT"), width=15, highlightbackground=self.bg_color).pack(side="left", expand=True, padx=2)
        
        self.launch_btn = tk.Button(btn_container, text="Launch", command=lambda: self.finish("LAUNCH"), 
                                   width=btn_width, bg="#007AFF", fg="white", font=("Helvetica", 11, "bold"), highlightbackground=self.bg_color)
        self.launch_btn.pack(side="left", expand=True, padx=2)

    def check_status(self):
        def check_cmd(cmd):
            try:
                subprocess.run(cmd, shell=True, capture_output=True, check=True)
                return True
            except: return False

        self.set_stat("Homebrew", check_cmd("brew --version"))
        self.set_stat("Wine", check_cmd("wine --version"))
        self.set_stat("Winetricks", check_cmd("winetricks --version"))
        
        # QuickTime
        qt_ok = os.path.exists(os.path.expanduser("~/.wine/drive_c/windows/system32/qtmlClient.dll")) or \
                os.path.exists(os.path.expanduser("~/.wine/drive_c/Program Files/QuickTime"))
        self.set_stat("QuickTime", qt_ok)

    def set_stat(self, item, is_ok):
        text = "✅ Installed" if is_ok else "❌ Missing"
        color = "#34C759" if is_ok else "#FF3B30"
        self.status_vars[item].config(text=text, fg=color)

    def finish(self, action):
        print(f"ACTION:{action}")
        self.root.destroy()

if __name__ == "__main__":
    res_dir = sys.argv[1] if len(sys.argv) > 1 else "."
    app = Director64Dashboard(res_dir)
    app.root.mainloop()
